#ifndef _SQLEXPORT_CPP
#define _SQLEXPORT_CPP
///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#include <Windows.H>
#include <WindowsX.H>
#include <ShellAPI.H>
#include <Stdio.H>
#include <Stdlib.H>
#include <SQL.H>
#include <SQLExt.H>

///////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

#include "../Resources/Resource.H"
#include "../../SharedClasses/CRC32/CRC.H"

#include "../CSockSrvr/CSockSrvr.H"

#include "../../SharedSource/NSWFL.H"
#include "Init.H"
#include "Entry.H"
#include "Routines.H"
#include "Command.H"
#include "SQLExport.H"
#include "Console.H"

//////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

void WriteNull(FILE *fTarget)
{
	const char *sNullStr = "NULL";
	const int iNullStrSz = 4;

	fwrite(&iNullStrSz, sizeof(iNullStrSz), 1, fTarget);
    fwrite(sNullStr, sizeof(char), iNullStrSz, fTarget);
}

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int ReplaceSingleQuotes(const char *sBuf, char *sOutBuf, int iBufSz)
{
	int iRPos = 0;
	int iWPos = 0;

	while(iRPos < iBufSz)
	{
		if(sBuf[iRPos] == '\'')
		{
			sOutBuf[iWPos++] = '\'';
		}

		sOutBuf[iWPos++] = sBuf[iRPos++];
	}

	sOutBuf[iWPos] = '\0';

	return iWPos;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////

int ExportSQLResults(CSockSrvr *pSockSrvr, int iClient, char *Statement,
					 char *sDB, char *sDBO, char *sImportTable, char *OutFileName)
{
	FILE *TargetHandle = NULL;

	CRecordSet rsTemp;

    SQLRETURN SQLResult = 0;

    char sStatusText[MAX_STATUS_TEXT]; //Just some RAM for status text and error messages.
    char sAltTemp[MAX_STATUS_TEXT];    //Just some RAM for status text and error messages.
    char sOutColumnName[256];

    int iInColumnNameLen = 256;
	int *iOutColumnSize = 0;
    int iOutColumnNameLen = 0;
    int *iOutDataType = 0;
    int iOutDecimalDigits = 0;
    int iOutNullable = 0;
    int iColumn = 0;

	rsTemp.bReplaceSingleQuotes = false;
	rsTemp.bThrowErrors = true;
	rsTemp.bTrimCharData = true;

	if(gbDebugMode)
	{
		WriteLog(pSockSrvr->icClientID[iClient], "Executing SQL statement.");
	}

	if(!CCI.cCustSQL.Execute(Statement, &rsTemp))
    {
        sprintf(sStatusText, "ExportSQLResults SELECT Execute: Fail on (%s)", Statement);

		sprintf(sStatusText, "Statement: %s", Statement);
		WriteLog(pSockSrvr->icClientID[iClient], sStatusText);

		//Need to do some advanced error reporting.
		if(CCI.cCustSQL.GetErrorMessage(sAltTemp, sizeof(sAltTemp), rsTemp.hSTMT))
		{
			sprintf(sStatusText, "Error Msg: %s", sAltTemp);
			WriteLog(pSockSrvr->icClientID[iClient], sStatusText);
		}

		WriteLog(pSockSrvr->icClientID[iClient], sStatusText);
		giWarningCount++;
        return SQL_IMPORT_RESULT_ERROR;
    }

    int iOutColCount = rsTemp.ColCount;
    int iOutRowCount = rsTemp.RowCount;

    if(iOutRowCount == 0)
    {
		if(gbDebugMode)
		{
			WriteLog(pSockSrvr->icClientID[iClient], "Query returned zero rows.");
		}

        rsTemp.Close();
        return SQL_IMPORT_RESULT_ZEROROWS;
    }
    else{
        sprintf(sStatusText, "Exporting transaction data. [%s - %d rows, %d columns]", sImportTable, iOutRowCount, iOutColCount);
		WriteLog(pSockSrvr->icClientID[iClient], sStatusText);
    }

    if((TargetHandle = fopen(OutFileName, "wb")) == NULL)
    {
        sprintf(sStatusText, "TargetFile: Binary write error with (%s).", OutFileName);
		WriteLog(pSockSrvr->icClientID[iClient], sStatusText);
		giWarningCount++;

        rsTemp.Close();
        return SQL_IMPORT_RESULT_ERROR;
    }

	if(gbDebugMode)
	{
		WriteLog(pSockSrvr->icClientID[iClient], "Writing export file header.");
	}

	int iImportTableNameLen = strlen(sImportTable);
    fwrite(&iImportTableNameLen, sizeof(iImportTableNameLen), 1, TargetHandle); // Write import table name len.
    fwrite(sImportTable, sizeof(char), iImportTableNameLen, TargetHandle);      // Write import table name.
    fwrite(&iOutColCount, sizeof(iOutColCount), 1, TargetHandle);               // Write number of columns.
    fwrite(&iOutRowCount, sizeof(iOutRowCount), 1, TargetHandle);               // Write number of rows.

    //---------------( Get column Information )---

	if(gbDebugMode)
	{
		WriteLog(pSockSrvr->icClientID[iClient], "Retreving column set information.");
	}

	iOutDataType = (int *) calloc( sizeof(int), iOutColCount);
	iOutColumnSize = (int *) calloc( sizeof(int), iOutColCount);

    while(rsTemp.GetColInfo(iColumn + 1, sOutColumnName, iInColumnNameLen, &iOutColumnNameLen,
		&iOutDataType[iColumn], &iOutColumnSize[iColumn], &iOutDecimalDigits, &iOutNullable))
	{
		fwrite(&iOutDataType[iColumn], sizeof(iOutDataType[iColumn]), 1, TargetHandle);
		fwrite(&iOutColumnSize[iColumn], sizeof(iOutColumnSize[iColumn]), 1, TargetHandle);
		fwrite(&iOutColumnNameLen, sizeof(iOutColumnNameLen), 1, TargetHandle);
		fwrite(sOutColumnName, sizeof(char), iOutColumnNameLen, TargetHandle);
		iColumn++;
    }

	if(gbDebugMode)
	{
		WriteLog(pSockSrvr->icClientID[iClient], "Exporting column set data.");
	}

	char *sSQLBuf = (char *) calloc(sizeof(char), (MAX_SQL_DATA_SZ * 2) + 1);
	char *sSQLBuf2 = (char *) calloc(sizeof(char), (MAX_SQL_DATA_SZ * 2) + 1);

	//---------------( Get column Data )---
    while(rsTemp.Fetch())
    {
        iColumn = 0;

        while(iColumn < iOutColCount)
        {
            int iDataType = iOutDataType[iColumn];
            int iDataLen = iOutColumnSize[iColumn];

            if(iDataType == SRV_TDS_IMAGE)
			{
				//Unsupported.
				WriteNull(TargetHandle);
			}
			else if(iDataType == SRV_TDS_CHAR || iDataType == SRV_TDS_VARCHAR ||
				iDataType == SRV_TDS_LONGVARCHAR || iDataType == SRV_TDS_NTEXT ||
				iDataType == SRV_TDS_UID || iDataType == SRV_TDS_NVARCHAR)
            {
                //If the data type is NText then only retreve a few KB's of it.
				if(iDataType == SRV_TDS_NTEXT)
				{
					iDataLen = MAX_SQL_DATA_SZ;
				}

				if(rsTemp.sColumnEx(iColumn + 1, sSQLBuf, iDataLen + 100, &iDataLen))
                {
					if(iDataType == SRV_TDS_NTEXT)
					{
						if(iDataLen > MAX_SQL_DATA_SZ)
						{
							iDataLen = MAX_SQL_DATA_SZ;
						}
					}

                    if(iDataLen == -1)
                    {
                        WriteNull(TargetHandle);
                    }
                    else{
						sSQLBuf[iDataLen] = '\0';

						iDataLen = ReplaceSingleQuotes(sSQLBuf, sSQLBuf2, iDataLen);

                        iDataLen = (iDataLen + 2); // To make room for the single quotes
                        fwrite(&iDataLen, sizeof(iDataLen), 1, TargetHandle);
                        iDataLen = (iDataLen - 2); // Because we only write the origional trimmed length

                        fwrite("'", sizeof(char), 1, TargetHandle); // Wrap the data in single quotes
                        fwrite(sSQLBuf2, sizeof(char), iDataLen, TargetHandle);
                        fwrite("'", sizeof(char), 1, TargetHandle); // Wrap the data in single quotes
					}
				}
				else{
	                sprintf(sStatusText, "Failed to retreive data from column. Table: %s, Column: %d, Type: %d", sImportTable, iColumn, iDataType);
					WriteLog(pSockSrvr->icClientID[iClient], sStatusText);
					giWarningCount++;
				}
            }
            else if(iDataType == SRV_TDS_DATE || iDataType == SRV_TDS_TIME || iDataType == SRV_TDS_TIMESTAMP)
            {
	            if(rsTemp.sColumnEx(iColumn + 1, sSQLBuf, MAX_SQL_DATA_SZ, &iDataLen))
				{
                    if(iDataLen == -1)
                    {
                        WriteNull(TargetHandle);
                    }
                    else{
	                    sSQLBuf[iDataLen] = '\0';

                        iDataLen = (iDataLen + 2); // To make room for the single quotes
                        fwrite(&iDataLen, sizeof(iDataLen), 1, TargetHandle);
                        iDataLen = (iDataLen - 2); // Because we only write the origional trimmed length

                        fwrite("'", sizeof(char), 1, TargetHandle); // Wrap the data in single quotes
                        fwrite(sSQLBuf, sizeof(char), iDataLen, TargetHandle);
                        fwrite("'", sizeof(char), 1, TargetHandle); // Wrap the data in single quotes
                    }
                }
				else{
	                sprintf(sStatusText, "Failed to retreive data from column. Table: %s, Column: %d, Type: %d", sImportTable, iColumn, iDataType);
					WriteLog(pSockSrvr->icClientID[iClient], sStatusText);
					giWarningCount++;
				}
			}
            else if(iDataType == SRV_TDS_INTEGER || iDataType == SRV_TDS_DECIMAL ||
				iDataType == SRV_TDS_NUMERIC || iDataType == SRV_TDS_SMALLINT ||
				iDataType == SRV_TDS_BIT || iDataType == SRV_TDS_REAL)
            {
                SQLINTEGER SQLBuffer3 = 0;
                char sCharData[32];

                if(rsTemp.GetData(iColumn+1, SQL_C_LONG, &SQLBuffer3, 32, (SQLINTEGER *)&iDataLen))
				{
                    if(iDataLen == -1)
                    {
                        WriteNull(TargetHandle);
                    }
                    else{
                        sprintf(sCharData, "%d", SQLBuffer3);
                        iDataLen = strlen(sCharData);
                        fwrite(&iDataLen, sizeof(iDataLen), 1, TargetHandle);
                        fwrite(sCharData, sizeof(char), iDataLen, TargetHandle);
                    }
                }
				else{
	                sprintf(sStatusText, "Failed to retreive data from column. Table: %s, Column: %d, Type: %d", sImportTable, iColumn, iDataType);
					WriteLog(pSockSrvr->icClientID[iClient], sStatusText);
					giWarningCount++;
				}
            }
            else if(iDataType == SRV_TDS_DOUBLE || iDataType == SRV_TDS_FLOAT)
            {
                SQLDOUBLE SQLBuffer4 = 0;
                char sCharData[32];

                if(rsTemp.GetData(iColumn+1, SQL_C_DOUBLE, &SQLBuffer4, 32, (SQLINTEGER *)&iDataLen))
                {
                    if(iDataLen == -1)
                    {
                        WriteNull(TargetHandle);
                    }
                    else{
                        sprintf(sCharData, "%f", SQLBuffer4);
                        iDataLen = strlen(sCharData);
                        fwrite(&iDataLen, sizeof(iDataLen), 1, TargetHandle);
                        fwrite(sCharData, sizeof(char), iDataLen, TargetHandle);
                    }
                }
				else{
	                sprintf(sStatusText, "Failed to retreive data from column. Table: %s, Column: %d, Type: %d", sImportTable, iColumn, iDataType);
					WriteLog(pSockSrvr->icClientID[iClient], sStatusText);
					giWarningCount++;
				}
            }
			else{
                sprintf(sStatusText, "Unknown SQL data type. Table: %s, Select column: %d, Type: %d", sImportTable, iColumn, iDataType);
				WriteLog(pSockSrvr->icClientID[iClient], sStatusText);

				free(sSQLBuf);
				free(sSQLBuf2);
				free(iOutColumnSize);
				free(iOutDataType);

				rsTemp.Close();

				if(TargetHandle) fclose(TargetHandle);

				giWarningCount++;
			    return SQL_IMPORT_RESULT_ERROR;
			}

            iColumn++;
        }
    }

	if(gbDebugMode)
	{
		WriteLog(pSockSrvr->icClientID[iClient], "Ending the export successfully.");
	}

	free(sSQLBuf);
	free(sSQLBuf2);
	free(iOutColumnSize);
	free(iOutDataType);

	rsTemp.Close();

    if(TargetHandle) fclose(TargetHandle);

	return SQL_IMPORT_RESULT_OK;
}

////////////////////////////////////////////////////////////////////////////////////////////////////////////////////////
#endif
